/*:
 * @target MZ
 * @plugindesc v1.2  Parallax Drift – subtle camera‑breathing effect with plugin‑command control. 2025‑07‑17 ★ Parallax layer fixed (RMMZ 1.8+).
 * @author さつまいも
 *
 * @help ParallaxDrift.js
 * ---------------------------------------------------------------------------
 * ▼ Change Log
 * ---------------------------------------------------------------------------
 *  v1.2 (2025‑07‑17)  • NEW: Proper support for "遠景" (parallax) by extending
 *                       Game_Map.parallaxOx / Oy instead of touching
 *                       private vars. Pictures logic unchanged.
 *  v1.1 (2025‑07‑17)  • Fixed MV‑only call ($gameScreen.pictures).  
 *  v1.0 (2025‑07‑17)  • First release.
 * ---------------------------------------------------------------------------
 * ▼ Overview (unchanged)
 * ---------------------------------------------------------------------------
 *  – Start / stop drift via plugin commands
 *  – Targets: parallax | all pictures | selected picture IDs
 *  – Sinusoidal loop (ease‑in/out)
 *  – MIT licence
 * ---------------------------------------------------------------------------
 * @command StartDrift
 * @text Start Drift
 * @desc Begin a parallax‑drift effect.
 *
 * @arg target
 * @type select
 * @option all
 * @option parallax
 * @option pictures
 * @option ids
 * @default all
 * @text Target
 *
 * @arg ids
 * @type string
 * @default
 * @text Picture IDs
 *
 * @arg ampX
 * @type number
 * @default 2
 * @text Amplitude X
 *
 * @arg ampY
 * @type number
 * @default 0
 * @text Amplitude Y
 *
 * @arg speed
 * @type number
 * @default 120
 * @text Speed (frames)
 *
 * @command StopDrift
 * @text Stop Drift
 * @desc End the current drift effect and reset offsets.
 */

(() => {
 "use strict";

 const PLUGIN_NAME = "ParallaxDrift";

 // -------------------------------------------------------------------------
 // ■ Drift State
 // -------------------------------------------------------------------------
 const Drift = {
   active: false,
   ampX: 0,
   ampY: 0,
   speed: 120,
   time: 0,
   target: "all",      // 'all' | 'parallax' | 'pictures' | 'ids'
   ids: [],             // picture IDs when target === 'ids'
   curX: 0,
   curY: 0
 };

 // -------------------------------------------------------------------------
 // ■ Utilities
 // -------------------------------------------------------------------------
 function forEachPicture(cb) {
   const max = $gameScreen.maxPictures ? $gameScreen.maxPictures() : 100;
   for (let i = 1; i <= max; i++) {
     const p = $gameScreen.picture(i);
     if (p) cb(p, i);
   }
 }

 // -------------------------------------------------------------------------
 // ■ Plugin Commands
 // -------------------------------------------------------------------------
 PluginManager.registerCommand(PLUGIN_NAME, "StartDrift", args => {
   Drift.active  = true;
   Drift.ampX    = Number(args.ampX || 0);
   Drift.ampY    = Number(args.ampY || 0);
   Drift.speed   = Math.max(1, Number(args.speed || 120));
   Drift.target  = String(args.target || "all");
   Drift.ids     = (args.ids || "").split(",").map(e => Number(e.trim())).filter(e => e);
   Drift.time    = 0;
   Drift.curX    = Drift.curY = 0;
 });

 PluginManager.registerCommand(PLUGIN_NAME, "StopDrift", () => {
   resetOffsets();
   Drift.active = false;
 });

 function resetOffsets() {
   Drift.curX = Drift.curY = 0;

   // pictures
   forEachPicture(p => { p._driftOffsetX = 0; p._driftOffsetY = 0; });
 }

 // -------------------------------------------------------------------------
 // ■ Extend Game_Map.parallaxOx / Oy (handles 遠景)
 // -------------------------------------------------------------------------
 const _Game_Map_parallaxOx = Game_Map.prototype.parallaxOx;
 Game_Map.prototype.parallaxOx = function() {
   const base = _Game_Map_parallaxOx.call(this);
   const add  = Drift.active && (Drift.target === "parallax" || Drift.target === "all") ? Drift.curX : 0;
   return base + add;
 };

 const _Game_Map_parallaxOy = Game_Map.prototype.parallaxOy;
 Game_Map.prototype.parallaxOy = function() {
   const base = _Game_Map_parallaxOy.call(this);
   const add  = Drift.active && (Drift.target === "parallax" || Drift.target === "all") ? Drift.curY : 0;
   return base + add;
 };

 // -------------------------------------------------------------------------
 // ■ Spriteset_Map – update loop (pictures)
 // -------------------------------------------------------------------------
 const _Spriteset_Map_update = Spriteset_Map.prototype.update;
 Spriteset_Map.prototype.update = function() {
   _Spriteset_Map_update.call(this);
   updateDrift();
 };

 function updateDrift() {
   if (!Drift.active) return;

   Drift.time++;
   const rad = (Drift.time / Drift.speed) * Math.PI * 2;
   const offX = Math.sin(rad) * Drift.ampX;
   const offY = Math.cos(rad) * Drift.ampY;

   Drift.curX = offX;
   Drift.curY = offY;

   // pictures
   if (Drift.target !== "parallax") {
     forEachPicture((pic, id) => {
       const ok = Drift.target === "all" || Drift.target === "pictures" || (Drift.target === "ids" && Drift.ids.includes(id));
       pic._driftOffsetX = ok ? offX : 0;
       pic._driftOffsetY = ok ? offY : 0;
     });
   }
 }

 // -------------------------------------------------------------------------
 // ■ Sprite_Picture – apply offsets
 // -------------------------------------------------------------------------
 const _Sprite_Picture_updatePosition = Sprite_Picture.prototype.updatePosition;
 Sprite_Picture.prototype.updatePosition = function() {
   _Sprite_Picture_updatePosition.call(this);
   const pic = this.picture();
   if (pic) {
     this.x += pic._driftOffsetX || 0;
     this.y += pic._driftOffsetY || 0;
   }
 };

})();

